<?php
declare(strict_types = 1);

/**
 * Artificial neural network for PHP
 *
 * @link https://ann.thwien.de/
 * @author Thomas Wien
 * @version 3.0
 * @copyright Copyright 2007-2025 by Thomas Wien
 * @license https://opensource.org/license/BSD-2-Clause BSD 2-Clause License
 */
namespace ANN;

/**
 *
 * @access private
 */
final class Neuron
{
    use Maths;

    /**
     *
     * @var array
     */
    protected ?array $arrInputs = null;

    /**
     *
     * @var array
     */
    protected ?array $arrWeights = null;

    /**
     *
     * @var float
     */
    protected ?float $floatOutput = null;

    /**
     *
     * @var float
     */
    protected float $floatDelta = 0;

    /**
     *
     * @var Network
     */
    protected ?Network $objNetwork = null;

    protected float $floatLearningRate = 0;

    /**
     *
     * @param Network $objNetwork
     * @uses Maths::randomDelta()
     */
    public function __construct(Network $objNetwork)
    {
        $this->objNetwork = $objNetwork;

        $this->floatDelta = $this->randomDelta();

        $this->floatLearningRate = $this->objNetwork->floatLearningRate;
    }

    /**
     *
     * @param
     *            array &$arrInputs
     * @uses initializeWeights()
     */
    public function setInputs(array &$arrInputs): void
    {
        $this->arrInputs = $arrInputs;

        $this->arrInputs[] = 1; // bias

        if ($this->arrWeights === null)
            $this->initializeWeights();
    }

    /**
     *
     * @param float $floatDelta
     */
    public function setDelta(float $floatDelta): void
    {
        $this->floatDelta = $floatDelta;
    }

    /**
     *
     * @return array
     */
    public function getWeights(): array
    {
        return $this->arrWeights;
    }

    /**
     *
     * @param integer $intKeyNeuron
     * @return float
     */
    public function getWeight(int $intKeyNeuron): float
    {
        return $this->arrWeights[$intKeyNeuron];
    }

    /**
     *
     * @return float
     */
    public function getOutput(): float
    {
        return $this->floatOutput;
    }

    /**
     *
     * @return float
     */
    public function getDelta(): float
    {
        return $this->floatDelta;
    }

    /**
     *
     * @uses Maths::randomWeight()
     */
    protected function initializeWeights(): void
    {
        foreach ($this->arrInputs as $intKey => $floatInput)
            $this->arrWeights[$intKey] = $this->randomWeight();
    }

    /**
     *
     * @uses Maths::sigmoid()
     */
    public function activate(): void
    {
        $floatSum = 0;

        foreach ($this->arrInputs as $intKey => $floatInput)
            $floatSum += $floatInput * $this->arrWeights[$intKey];

        $this->floatOutput = $this->sigmoid($floatSum);
    }

    public function adjustWeights(): void
    {
        $floatLearningRateDeltaFactor = $this->floatLearningRate * $this->floatDelta;

        foreach ($this->arrWeights as $intKey => $floatWeight)
            $this->arrWeights[$intKey] += $this->arrInputs[$intKey] * $floatLearningRateDeltaFactor;
    }
}
