<?php
declare(strict_types = 1);

/**
 * Artificial neural network for PHP
 *
 * @link https://ann.thwien.de/
 * @author Thomas Wien
 * @version 3.0
 * @copyright Copyright 2007-2025 by Thomas Wien
 * @license https://opensource.org/license/BSD-2-Clause BSD 2-Clause License
 */
namespace ANN;

class Network extends Filesystem implements InterfaceLoadable
{

    /**
     *
     * @var Layer
     */
    protected ?Layer $objOutputLayer = null;

    /**
     *
     * @var array
     */
    protected array $arrHiddenLayers = array();

    /**
     *
     * @var array
     */
    protected ?array $arrInputs = null;

    /**
     *
     * @var array
     */
    protected ?array $arrOutputs = null;

    /**
     *
     * @var integer
     */
    protected int $intTotalLoops = 0;

    /**
     *
     * @var integer
     */
    protected int $intTotalTrainings = 0;

    /**
     *
     * @var integer
     */
    protected int $intTotalActivations = 0;

    /**
     *
     * @var integer
     */
    protected int $intTotalActivationsRequests = 0;

    /**
     *
     * @var integer
     */
    protected ?int $intNumberOfHiddenLayers = null;

    /**
     *
     * @var integer
     */
    protected ?int $intNumberOfHiddenLayersDec = null;

    // decremented value

    /**
     *
     * @var integer
     */
    protected int $intNumberEpoch = 0;

    /**
     *
     * @var boolean
     */
    protected bool $boolTrained = FALSE;

    /**
     *
     * @var integer
     */
    protected int $intTrainingTime = 0;

    // Seconds

    /**
     *
     * @var boolean
     */
    protected bool $boolNetworkActivated = FALSE;

    /**
     *
     * @var array
     */
    protected array $arrTrainingComplete = array();

    /**
     *
     * @var integer
     */
    protected int $intNumberOfNeuronsPerLayer = 0;

    /**
     *
     * @var float
     */
    protected float $floatOutputErrorTolerance = 0.02;

    /**
     *
     * @var float
     */
    public float $floatMomentum = 0.95;

    /**
     *
     * @var array
     */
    private array $arrInputsToTrain = array();

    /**
     *
     * @var integer
     */
    private int $intInputsToTrainIndex = - 1;

    /**
     *
     * @var integer
     */
    public int $intOutputType = self::OUTPUT_LINEAR;

    /**
     *
     * @var float
     */
    public float $floatLearningRate = 0.7;

    /**
     *
     * @var boolean
     */
    public bool $boolFirstLoopOfTraining = TRUE;

    /**
     *
     * @var boolean
     */
    public bool $boolFirstEpochOfTraining = TRUE;

    /**
     * Linear output type
     */
    const OUTPUT_LINEAR = 1;

    /**
     * Binary output type
     */
    const OUTPUT_BINARY = 2;

    /**
     *
     * @param integer $intNumberOfHiddenLayers
     *            (Default: 1)
     * @param integer $intNumberOfNeuronsPerLayer
     *            (Default: 6)
     * @param integer $intNumberOfOutputs
     *            (Default: 1)
     * @uses Exception::__construct()
     * @uses createHiddenLayers()
     * @uses createOutputLayer()
     * @throws Exception
     */
    public function __construct(int $intNumberOfHiddenLayers = 1, int $intNumberOfNeuronsPerLayer = 6, int $intNumberOfOutputs = 1)
    {
        if ($intNumberOfHiddenLayers < 1)
            throw new Exception('Constraints: $intNumberOfHiddenLayers must be a positiv integer >= 1');

        if ($intNumberOfNeuronsPerLayer < 2)
            throw new Exception('Constraints: $intNumberOfNeuronsPerLayer must be a positiv integer number >= 2');

        if ($intNumberOfOutputs < 1)
            throw new Exception('Constraints: $intNumberOfOutputs must be a positiv integer number >= 1');

        $this->createOutputLayer($intNumberOfOutputs);

        $this->createHiddenLayers($intNumberOfHiddenLayers, $intNumberOfNeuronsPerLayer);

        $this->intNumberOfHiddenLayers = $intNumberOfHiddenLayers;

        $this->intNumberOfHiddenLayersDec = $this->intNumberOfHiddenLayers - 1;

        $this->intNumberOfNeuronsPerLayer = $intNumberOfNeuronsPerLayer;
    }

    /**
     *
     * @param array $arrInputs
     */
    protected function setInputs(array $arrInputs): void
    {
        $this->arrInputs = $arrInputs;

        $this->intNumberEpoch = count($arrInputs);

        $this->nextIndexInputToTrain = 0;

        $this->boolNetworkActivated = FALSE;
    }

    /**
     *
     * @param array $arrOutputs
     * @uses Exception::__construct()
     * @uses Layer::getNeuronsCount()
     * @throws Exception
     */
    protected function setOutputs(array $arrOutputs): void
    {
        if (isset($arrOutputs[0]) && is_array($arrOutputs[0]))
            if (count($arrOutputs[0]) != $this->objOutputLayer->getNeuronsCount())
                throw new Exception('Count of arrOutputs doesn\'t fit to number of arrOutputs on instantiation of \\' . __NAMESPACE__ . '\\Network');

        $this->arrOutputs = $arrOutputs;

        $this->boolNetworkActivated = FALSE;
    }

    /**
     * Set Values for training or using network
     *
     * Set Values of inputs and outputs for training or just inputs for using
     * already trained network.
     *
     * <code>
     * $objNetwork = new \ANN\Network(2, 4, 1);
     *
     * $objValues = new \ANN\Values;
     *
     * $objValues->train()
     * ->input(0.12, 0.11, 0.15)
     * ->output(0.56);
     *
     * $objNetwork->setValues($objValues);
     * </code>
     *
     * @param Values $objValues
     * @uses Values::getInputsArray()
     * @uses Values::getOutputsArray()
     * @uses setInputs()
     * @uses setOutputs()
     */
    public function setValuesToTrain(Values $objValues): void
    {
        $this->setInputs($objValues->getInputsArray());

        $this->setOutputs($objValues->getOutputsArray());
    }

    /**
     * Set Values for training or using network
     *
     * Set Values of inputs and outputs for training or just inputs for using
     * already trained network.
     *
     * <code>
     * $objNetwork = new \ANN\Network(2, 4, 1);
     *
     * $objValues = new \ANN\Values;
     *
     * $objValues->train()
     * ->input(0.12, 0.11, 0.15)
     * ->output(0.56);
     *
     * $objNetwork->setValues($objValues);
     * </code>
     *
     * @param Values $objValues
     * @uses Values::getInputsArray()
     * @uses Values::getOutputsArray()
     * @uses setInputs()
     * @uses setOutputs()
     */
    public function setValues(Values $objValues): void
    {
        $this->setInputs($objValues->getInputsArray());

        $this->setOutputs($objValues->getOutputsArray());
    }

    /**
     *
     * @param array $arrInputs
     * @uses Layer::setInputs()
     */
    protected function setInputsToTrain(array $arrInputs): void
    {
        $this->arrHiddenLayers[0]->setInputs($arrInputs);

        $this->boolNetworkActivated = FALSE;
    }

    /**
     * Get the output values
     *
     * Get the output values to the related input values set by setValues(). This
     * method returns the output values as a two-dimensional array.
     *
     * @return array two-dimensional array
     * @uses activate()
     * @uses getCountInputs()
     * @uses Layer::getOutputs()
     * @uses Layer::getThresholdOutputs()
     * @uses setInputsToTrain()
     */
    public function getOutputs(): array
    {
        $arrReturnOutputs = array();

        $intCountInputs = $this->getCountInputs();

        for ($intIndex = 0; $intIndex < $intCountInputs; $intIndex ++)
        {
            $this->setInputsToTrain($this->arrInputs[$intIndex]);

            $this->activate();

            switch ($this->intOutputType)
            {
                case self::OUTPUT_LINEAR:
                    $arrReturnOutputs[] = $this->objOutputLayer->getOutputs();
                    break;

                case self::OUTPUT_BINARY:
                    $arrReturnOutputs[] = $this->objOutputLayer->getThresholdOutputs();
                    break;
            }
        }

        return $arrReturnOutputs;
    }

    /**
     *
     * @param integer $intKeyInput
     * @return array
     * @uses activate()
     * @uses Layer::getOutputs()
     * @uses Layer::getThresholdOutputs()
     * @uses setInputsToTrain()
     */
    public function getOutputsByInputKey(int $intKeyInput): array
    {
        $this->setInputsToTrain($this->arrInputs[$intKeyInput]);

        $this->activate();

        switch ($this->intOutputType)
        {
            case self::OUTPUT_LINEAR:
                return $this->objOutputLayer->getOutputs();

            case self::OUTPUT_BINARY:
                return $this->objOutputLayer->getThresholdOutputs();
        }
    }

    /**
     *
     * @param integer $intNumberOfHiddenLayers
     * @param integer $intNumberOfNeuronsPerLayer
     * @uses Layer::__construct()
     */
    protected function createHiddenLayers(int $intNumberOfHiddenLayers, int $intNumberOfNeuronsPerLayer): void
    {
        $layerId = $intNumberOfHiddenLayers;

        for ($i = 0; $i < $intNumberOfHiddenLayers; $i ++)
        {
            $layerId --;

            if ($i == 0)
                $nextLayer = $this->objOutputLayer;

            if ($i > 0)
                $nextLayer = $this->arrHiddenLayers[$layerId + 1];

            $this->arrHiddenLayers[$layerId] = new Layer($this, $intNumberOfNeuronsPerLayer, $nextLayer);
        }

        ksort($this->arrHiddenLayers);
    }

    /**
     *
     * @param integer $intNumberOfOutputs
     * @uses Layer::__construct()
     */
    protected function createOutputLayer(int $intNumberOfOutputs): void
    {
        $this->objOutputLayer = new Layer($this, $intNumberOfOutputs);
    }

    /**
     *
     * @uses Layer::setInputs()
     * @uses Layer::activate()
     * @uses Layer::getOutputs()
     */
    protected function activate(): void
    {
        $this->intTotalActivationsRequests ++;

        if ($this->boolNetworkActivated)
            return;

        $this->arrHiddenLayers[0]->activate();

        $this->boolNetworkActivated = TRUE;

        $this->intTotalActivations ++;
    }

    /**
     *
     * @return boolean
     * @uses Exception::__construct()
     * @uses setInputs()
     * @uses setOutputs()
     * @uses isTrainingComplete()
     * @uses isTrainingCompleteByEpoch()
     * @uses setInputsToTrain()
     * @uses training()
     * @uses isEpoch()
     * @uses logWeights()
     * @uses logNetworkErrors()
     * @uses getNextIndexInputsToTrain()
     * @uses isTrainingCompleteByInputKey()
     * @uses setDynamicLearningRate()
     * @uses detectOutputType()
     * @throws Exception
     */
    public function train(): bool
    {
        if (! $this->arrInputs)
            throw new Exception('No arrInputs defined. Use \\' . __NAMESPACE__ . '\\Network::setValues().');

        if (! $this->arrOutputs)
            throw new Exception('No arrOutputs defined. Use \\' . __NAMESPACE__ . '\\Network::setValues().');

        $this->detectOutputType();

        if ($this->isTrainingComplete())
        {
            $this->boolTrained = TRUE;

            return $this->boolTrained;
        }

        $intStartTime = date('U');

        $this->getNextIndexInputsToTrain(TRUE);

        $this->boolFirstLoopOfTraining = TRUE;

        $this->boolFirstEpochOfTraining = TRUE;

        $intLoop = 0;

        while (TRUE)
        {
            $intLoop ++;

            $this->setDynamicLearningRate($intLoop);

            $j = $this->getNextIndexInputsToTrain();

            $this->setInputsToTrain($this->arrInputs[$j]);

            if (! ($this->arrTrainingComplete[$j] = $this->isTrainingCompleteByInputKey($j)))
                $this->training($this->arrOutputs[$j]);

            if ($this->isEpoch())
            {
                if ($this->isTrainingCompleteByEpoch())
                    break;

                $this->boolFirstEpochOfTraining = FALSE;
            }

            $this->boolFirstLoopOfTraining = FALSE;
        }

        $intStopTime = date('U');

        $this->intTotalLoops += $intLoop;

        $this->intTrainingTime += $intStopTime - $intStartTime;

        $this->boolTrained = $this->isTrainingComplete();

        return $this->boolTrained;
    }

    /**
     *
     * @param boolean $boolReset
     *            (Default: FALSE)
     * @return integer|null
     */
    protected function getNextIndexInputsToTrain(bool $boolReset = FALSE): ?int
    {
        if ($boolReset)
        {
            $this->arrInputsToTrain = array_keys($this->arrInputs);

            $this->intInputsToTrainIndex = - 1;

            return null;
        }

        $this->intInputsToTrainIndex ++;

        if (! isset($this->arrInputsToTrain[$this->intInputsToTrainIndex]))
        {
            shuffle($this->arrInputsToTrain);

            $this->intInputsToTrainIndex = 0;
        }

        return $this->arrInputsToTrain[$this->intInputsToTrainIndex];
    }

    /**
     *
     * @return integer
     */
    public function getTotalLoops(): int
    {
        return $this->intTotalLoops;
    }

    /**
     *
     * @return boolean
     */
    protected function isEpoch(): bool
    {
        static $countLoop = 0;

        $countLoop ++;

        if ($countLoop >= $this->intNumberEpoch)
        {
            $countLoop = 0;

            return TRUE;
        }

        return FALSE;
    }

    /**
     * Setting the learning rate
     *
     * @param float $floatLearningRate
     *            (Default: 0.7) (0.1 .. 0.9)
     * @uses Exception::__construct()
     * @throws Exception
     */
    protected function setLearningRate(float $floatLearningRate = 0.7): void
    {
        if (! is_float($floatLearningRate))
            throw new Exception('$floatLearningRate should be between 0.1 and 0.9');

        if ($floatLearningRate <= 0 || $floatLearningRate >= 1)
            throw new Exception('$floatLearningRate should be between 0.1 and 0.9');

        $this->floatLearningRate = $floatLearningRate;
    }

    /**
     *
     * @return boolean
     * @uses getOutputs()
     */
    protected function isTrainingComplete(): bool
    {
        $arrOutputs = $this->getOutputs();

        switch ($this->intOutputType)
        {
            case self::OUTPUT_LINEAR:

                foreach ($this->arrOutputs as $intKey1 => $arrOutput)
                    foreach ($arrOutput as $intKey2 => $floatValue)
                        if (($floatValue > round($arrOutputs[$intKey1][$intKey2] + $this->floatOutputErrorTolerance, 3)) || ($floatValue < round($arrOutputs[$intKey1][$intKey2] - $this->floatOutputErrorTolerance, 3)))
                            return FALSE;

                return TRUE;

            case self::OUTPUT_BINARY:

                foreach ($this->arrOutputs as $intKey1 => $arrOutput)
                    foreach ($arrOutput as $intKey2 => $floatValue)
                        if ($floatValue != $arrOutputs[$intKey1][$intKey2])
                            return FALSE;

                return TRUE;
        }
    }

    /**
     *
     * @return boolean
     */
    protected function isTrainingCompleteByEpoch(): bool
    {
        foreach ($this->arrTrainingComplete as $trainingComplete)
            if (! $trainingComplete)
                return FALSE;

        return TRUE;
    }

    /**
     *
     * @param integer $intKeyInput
     * @return boolean
     * @uses getOutputsByInputKey()
     */
    protected function isTrainingCompleteByInputKey(int $intKeyInput): bool
    {
        $arrOutputs = $this->getOutputsByInputKey($intKeyInput);

        if (! isset($this->arrOutputs[$intKeyInput]))
            return TRUE;

        switch ($this->intOutputType)
        {
            case self::OUTPUT_LINEAR:

                foreach ($this->arrOutputs[$intKeyInput] as $intKey => $floatValue)
                    if (($floatValue > round($arrOutputs[$intKey] + $this->floatOutputErrorTolerance, 3)) || ($floatValue < round($arrOutputs[$intKey] - $this->floatOutputErrorTolerance, 3)))
                        return FALSE;

                return TRUE;

            case self::OUTPUT_BINARY:

                foreach ($this->arrOutputs[$intKeyInput] as $intKey => $floatValue)
                    if ($floatValue != $arrOutputs[$intKey])
                        return FALSE;

                return TRUE;
        }
    }

    /**
     *
     * @return integer
     */
    protected function getCountInputs(): int
    {
        if (isset($this->arrInputs) && is_array($this->arrInputs))
            return count($this->arrInputs);

        return 0;
    }

    /**
     *
     * @param array $arrOutputs
     * @uses activate()
     * @uses Layer::calculateHiddenDeltas()
     * @uses Layer::adjustWeights()
     * @uses Layer::calculateOutputDeltas()
     * @uses getNetworkError()
     */
    protected function training(array $arrOutputs): void
    {
        $this->activate();

        $this->objOutputLayer->calculateOutputDeltas($arrOutputs);

        for ($i = $this->intNumberOfHiddenLayersDec; $i >= 0; $i --)
            $this->arrHiddenLayers[$i]->calculateHiddenDeltas();

        $this->objOutputLayer->adjustWeights();

        for ($i = $this->intNumberOfHiddenLayersDec; $i >= 0; $i --)
            $this->arrHiddenLayers[$i]->adjustWeights();

        $this->intTotalTrainings ++;

        $this->boolNetworkActivated = FALSE;
    }

    /**
     *
     * @param integer $intType
     *            (Default: Network::OUTPUT_LINEAR)
     * @uses Exception::__construct()
     * @throws Exception
     */
    protected function setOutputType(int $intType = self::OUTPUT_LINEAR): void
    {
        switch ($intType)
        {
            case self::OUTPUT_LINEAR:
            case self::OUTPUT_BINARY:
                $this->intOutputType = $intType;
                break;

            default:
                throw new Exception('$strType must be \\' . __NAMESPACE__ . '\\Network::OUTPUT_LINEAR or \\' . __NAMESPACE__ . '\\Network::OUTPUT_BINARY');
        }
    }

    public function __wakeup(): void
    {
        $this->boolNetworkActivated = FALSE;
    }

    /**
     *
     * @param string $strFilename
     * @return Network
     * @uses parent::loadFromFile()
     */
    public static function loadFromFile(string $strFilename): Network
    {
        return parent::loadFromFile($strFilename);
    }

    /**
     *
     * @param string $strFilename
     *            (Default: null)
     * @uses parent::saveToFile()
     * @uses getDefaultFilename()
     */
    public function saveToFile(?string $strFilename = null): void
    {
        if ($strFilename === null)
            $strFilename = self::getDefaultFilename();

        parent::saveToFile($strFilename);
    }

    /**
     *
     * @return integer
     */
    public function getNumberInputs(): int
    {
        if (isset($this->arrInputs) && is_array($this->arrInputs))
            if (isset($this->arrInputs[0]))
                return count($this->arrInputs[0]);

        return 0;
    }

    /**
     *
     * @return integer
     */
    public function getNumberHiddenLayers(): int
    {
        if (isset($this->arrHiddenLayers) && is_array($this->arrHiddenLayers))
            return count($this->arrHiddenLayers);

        return 0;
    }

    /**
     *
     * @return integer
     */
    public function getNumberHiddens(): int
    {
        if (isset($this->arrHiddenLayers) && is_array($this->arrHiddenLayers))
            if (isset($this->arrHiddenLayers[0]))
                return $this->arrHiddenLayers[0]->getNeuronsCount();

        return 0;
    }

    /**
     *
     * @return integer
     */
    public function getNumberOutputs(): int
    {
        if (isset($this->arrOutputs[0]) && is_array($this->arrOutputs[0]))
            return count($this->arrOutputs[0]);

        return 0;
    }

    /**
     *
     * @return float
     * @uses getOutputs()
     */
    protected function getNetworkError(): float
    {
        $floatError = 0;

        $arrNetworkOutputs = $this->getOutputs();

        foreach ($this->arrOutputs as $intKeyOutputs => $arrDesiredOutputs)
            foreach ($arrDesiredOutputs as $intKeyOutput => $floatDesiredOutput)
                $floatError += pow($arrNetworkOutputs[$intKeyOutputs][$intKeyOutput] - $floatDesiredOutput, 2);

        return $floatError / 2;
    }

    /**
     *
     * @uses setOutputType()
     */
    protected function detectOutputType(): void
    {
        if (empty($this->arrOutputs))
            return;

        foreach ($this->arrOutputs as $arrOutputs)
            foreach ($arrOutputs as $floatOutput)
                if ($floatOutput < 1 && $floatOutput > 0)
                {
                    $this->setOutputType(self::OUTPUT_LINEAR);

                    return;
                }

        $this->setOutputType(self::OUTPUT_BINARY);
    }

    /**
     * Setting the percentage of output error in comparison to the desired output
     *
     * @param float $floatOutputErrorTolerance
     *            (Default: 0.02)
     */
    public function setOutputErrorTolerance(float $floatOutputErrorTolerance = 0.02): void
    {
        if ($floatOutputErrorTolerance < 0 || $floatOutputErrorTolerance > 0.1)
            throw new Exception('$floatOutputErrorTolerance must be between 0 and 0.1');

        $this->floatOutputErrorTolerance = $floatOutputErrorTolerance;
    }

    /**
     *
     * @param float $floatMomentum
     *            (Default: 0.95) (0 .. 1)
     * @uses Exception::__construct()
     * @throws Exception
     */
    public function setMomentum(float $floatMomentum = 0.95): void
    {
        if (! is_float($floatMomentum) && ! is_integer($floatMomentum))
            throw new Exception('$floatLearningRate should be between 0 and 1');

        if ($floatMomentum <= 0 || $floatMomentum > 1)
            throw new Exception('$floatLearningRate should be between 0 and 1');

        $this->floatMomentum = $floatMomentum;
    }

    /**
     * Dynamic Learning Rate
     *
     * Setting learning rate all 1000 loops dynamically
     *
     * @param integer $intLoop
     * @uses setLearningRate()
     */
    protected function setDynamicLearningRate(int $intLoop): void
    {
        if ($intLoop % 1000)
            return;

        $floatLearningRate = (mt_rand(5, 7) / 10);

        $this->setLearningRate($floatLearningRate);
    }

    /**
     *
     * @return array
     * @uses getNetworkError()
     * @uses getNumberInputs()
     * @uses getTrainedInputsPercentage()
     */
    public function getNetworkInfo(): array
    {
        $arrReturn = array();

        switch ($this->intOutputType)
        {
            case self::OUTPUT_BINARY:

                $arrReturn['detected_output_type'] = 'Binary';

                break;

            case self::OUTPUT_LINEAR:

                $arrReturn['detected_output_type'] = 'Linear';

                break;
        }

        $arrReturn['activation_function'] = 'Sigmoid';

        $arrReturn['momentum'] = $this->floatMomentum;

        $arrReturn['learning_rate'] = 'Dynamic';

        $arrReturn['network_error'] = $this->getNetworkError();

        $arrReturn['output_error_tolerance'] = $this->floatOutputErrorTolerance;

        $arrReturn['total_loops'] = $this->intTotalLoops;

        $arrReturn['total_trainings'] = $this->intTotalTrainings;

        $arrReturn['total_activations'] = $this->intTotalActivations;

        $arrReturn['total_activations_requests'] = $this->intTotalActivationsRequests;

        $arrReturn['epoch'] = $this->intNumberEpoch;

        $arrReturn['training_time_seconds'] = $this->intTrainingTime;

        $arrReturn['training_time_minutes'] = round($this->intTrainingTime / 60, 1);

        if ($this->intTrainingTime > 0)
        {
            $arrReturn['loops_per_second'] = round($this->intTotalLoops / $this->intTrainingTime);
        } else
        {
            $arrReturn['loops_per_second'] = round($this->intTotalLoops / 0.1);
        }

        $arrReturn['training_finished'] = ($this->boolTrained) ? 'Yes' : 'No';

        $arrReturn['network']['arrHiddenLayers'] = $this->arrHiddenLayers;

        $arrReturn['network']['objOutputLayer'] = $this->objOutputLayer;

        $arrReturn['network']['intCountInputs'] = $this->getNumberInputs();

        $arrReturn['trained_percentage'] = $this->getTrainedInputsPercentage();

        $arrReturn['phpversion'] = phpversion();

        $arrReturn['phpinterface'] = php_sapi_name();

        return $arrReturn;
    }

    /**
     *
     * @return float
     * @uses isTrainingCompleteByInputKey()
     */
    protected function getTrainedInputsPercentage(): float
    {
        $boolTrained = 0;

        $arrInputs = array();

        foreach ($this->arrInputs as $intKeyInputs => $arrInputs)
        {
            if ($this->isTrainingCompleteByInputKey($intKeyInputs))
                $boolTrained ++;
        }

        return round(($boolTrained / @count($this->arrOutputs)) * 100, 1);
    }
}
