<?php
declare(strict_types = 1);

/**
 * Artificial neural network for PHP
 *
 * @link https://ann.thwien.de/
 * @author Thomas Wien
 * @version 3.0
 * @copyright Copyright 2007-2025 by Thomas Wien
 * @license https://opensource.org/license/BSD-2-Clause BSD 2-Clause License
 */
namespace ANN;

/**
 *
 * @access private
 */
final class Layer
{
    use Maths;

    /**
     *
     * @var array
     */
    protected array $arrNeurons = array();

    /**
     *
     * @var array
     */
    protected array $arrOutputs = array();

    /**
     *
     * @var Network
     */
    protected ?Network $objNetwork = null;

    /**
     *
     * @var Layer
     */
    protected ?Layer $objNextLayer = null;

    /**
     *
     * @var integer
     */
    protected ?int $intNumberOfNeurons = null;

    /**
     *
     * @param Network $objNetwork
     * @param integer $intNumberOfNeurons
     * @param Layer $objNextLayer
     *            (Default: null)
     * @uses createNeurons()
     */
    public function __construct(Network $objNetwork, int $intNumberOfNeurons, ?Layer $objNextLayer = null)
    {
        $this->objNetwork = $objNetwork;

        $this->objNextLayer = $objNextLayer;

        $this->createNeurons($intNumberOfNeurons);

        $this->intNumberOfNeurons = $intNumberOfNeurons;
    }

    /**
     *
     * @param
     *            array &$arrInputs
     * @uses Neuron::setInputs()
     */
    public function setInputs(array &$arrInputs): void
    {
        foreach ($this->arrNeurons as $objNeuron)
            $objNeuron->setInputs($arrInputs);
    }

    /**
     *
     * @return array
     */
    public function getNeurons(): array
    {
        return $this->arrNeurons;
    }

    /**
     *
     * @return integer
     */
    public function getNeuronsCount(): int
    {
        return $this->intNumberOfNeurons;
    }

    /**
     *
     * @return array
     */
    public function getOutputs(): array
    {
        return $this->arrOutputs;
    }

    /**
     *
     * @return array
     * @uses Maths::threshold()
     */
    public function getThresholdOutputs(): array
    {
        $arrReturnOutputs = array();

        foreach ($this->arrOutputs as $intKey => $floatOutput)
            $arrReturnOutputs[$intKey] = $this->threshold($floatOutput);

        return $arrReturnOutputs;
    }

    /**
     *
     * @param integer $intNumberOfNeurons
     * @uses Neuron::__construct()
     */
    protected function createNeurons(int $intNumberOfNeurons): void
    {
        for ($intIndex = 0; $intIndex < $intNumberOfNeurons; $intIndex ++)
            $this->arrNeurons[] = new Neuron($this->objNetwork);
    }

    /**
     *
     * @uses Neuron::activate()
     * @uses Neuron::getOutput()
     * @uses Layer::setInputs()
     * @uses Layer::activate()
     */
    public function activate(): void
    {
        foreach ($this->arrNeurons as $intKey => $objNeuron)
        {
            $objNeuron->activate();

            $arrOutputs[$intKey] = $objNeuron->getOutput();
        }

        if ($this->objNextLayer !== null)
        {
            $this->objNextLayer->setInputs($arrOutputs);

            $this->objNextLayer->activate();
        }

        $this->arrOutputs = $arrOutputs;
    }

    /**
     *
     * @uses Neuron::setDelta()
     * @uses Neuron::getWeight()
     * @uses Neuron::getDelta()
     * @uses Neuron::getOutput()
     * @uses getNeurons()
     */
    public function calculateHiddenDeltas(): void
    {
        $floatDelta = 0;

        $floatSum = 0;

        $arrNeuronsNextLayer = $this->objNextLayer->getNeurons();

        /* @var $objNeuron Neuron */

        foreach ($this->arrNeurons as $intKeyNeuron => $objNeuron)
        {
            /* @var $objNeuronNextLayer Neuron */

            foreach ($arrNeuronsNextLayer as $objNeuronNextLayer)
                $floatSum += $objNeuronNextLayer->getWeight($intKeyNeuron) * $objNeuronNextLayer->getDelta() * $this->objNetwork->floatMomentum;

            $floatOutput = $objNeuron->getOutput();

            $floatDelta = $floatOutput * (1 - $floatOutput) * $floatSum;

            $objNeuron->setDelta($floatDelta);
        }
    }

    /**
     *
     * @param array $arrDesiredOutputs
     * @uses Neuron::setDelta()
     * @uses Neuron::getOutput()
     */
    public function calculateOutputDeltas(array $arrDesiredOutputs): void
    {
        foreach ($this->arrNeurons as $intKeyNeuron => $objNeuron)
        {
            $floatOutput = $objNeuron->getOutput();

            $floatDelta = $floatOutput * ($arrDesiredOutputs[$intKeyNeuron] - $floatOutput) * (1 - $floatOutput);

            $objNeuron->setDelta($floatDelta);
        }
    }

    /**
     *
     * @uses Neuron::adjustWeights()
     */
    public function adjustWeights(): void
    {
        foreach ($this->arrNeurons as $objNeuron)
            $objNeuron->adjustWeights();
    }
}
