<?php
declare(strict_types = 1);

/**
 * Artificial neural network for PHP
 *
 * @link https://ann.thwien.de/
 * @author Thomas Wien
 * @version 3.0
 * @copyright Copyright 2007-2025 by Thomas Wien
 * @license https://opensource.org/license/BSD-2-Clause BSD 2-Clause License
 */
namespace ANN;

final class InputValue extends Filesystem implements InterfaceLoadable
{

    /**
     *
     * @var float
     */
    protected float $floatMin = 0;

    /**
     *
     * @var float
     */
    protected float $floatMax = 0;

    /**
     *
     * @var boolean
     */
    protected bool $boolOverride = FALSE;

    /**
     *
     * @param float $floatMin
     * @param float $floatMax
     * @param boolean $boolOverride
     *            (Default: FALSE)
     * @throws Exception If $boolOverride is FALSE, an exception will be thrown if getInputValue() will
     *         be called with outranged values. If $boolOverride is TRUE, no exception will be
     *         thrown in this case, but lower values are replaced by $floatMin and upper values
     *         are replaced by $floatMax.
     */
    public function __construct(float $floatMin, float $floatMax, bool $boolOverride = FALSE)
    {
        if ($floatMin > $floatMax)
            throw new Exception('Constraints: $floatMin should be lower than $floatMax');

        $this->floatMin = $floatMin;

        $this->floatMax = $floatMax;

        $this->override = $boolOverride;
    }

    /**
     *
     * @param float $floatValue
     * @return float
     * @uses calculateInputValue()
     * @throws Exception
     */
    public function getInputValue(float $floatValue): float
    {
        if (! $this->override && $floatValue < $this->floatMin)
            throw new Exception('Constraints: $floatValue should be between ' . $this->floatMin . ' and ' . $this->floatMax);

        if (! $this->override && $floatValue > $this->floatMax)
            throw new Exception('Constraints: $floatValue should be between ' . $this->floatMin . ' and ' . $this->floatMax);

        if ($this->override && $floatValue < $this->floatMin)
            $floatValue = $this->floatMin;

        if ($this->override && $floatValue > $this->floatMax)
            $floatValue = $this->floatMax;

        if ($floatValue >= $this->floatMin && $floatValue <= $this->floatMax)
            return $this->calculateInputValue($floatValue);
    }

    /**
     *
     * @param float $floatValue
     * @return float
     */
    protected function calculateInputValue(float $floatValue): float
    {
        return ($floatValue - $this->floatMin) / ($this->floatMax - $this->floatMin);
    }
}
