<?php
declare(strict_types = 1);

/**
 * Artificial neural network for PHP
 *
 * @link https://ann.thwien.de/
 * @author Thomas Wien
 * @version 3.0
 * @copyright Copyright 2007-2025 by Thomas Wien
 * @license https://opensource.org/license/BSD-2-Clause BSD 2-Clause License
 */
namespace ANN;

/**
 *
 * @access private
 */
abstract class Filesystem
{

    /**
     *
     * @param string $strFilename
     * @uses Exception::__construct()
     * @throws Exception
     */
    public function saveToFile(string $strFilename): void
    {
        if (! ($this instanceof InterfaceLoadable))
            throw new Exception('Current object not instance of Interface \\ANN\\InterfaceLoadable');

        if (empty($strFilename))
            throw new Exception('Paramter $strFilename should be a filename');

        $strDir = dirname($strFilename);

        if (empty($strDir))
            $strDir = '.';

        if (! is_dir($strDir))
            throw new Exception("Directory $strDir does not exist");

        if (! is_writable($strDir))
            throw new Exception("Directory $strDir has no writing permission");

        if (is_file($strFilename) && ! is_writable($strFilename))
            throw new Exception("File $strFilename does exist but has no writing permission");

        try
        {
            $strSerialized = serialize($this);

            file_put_contents($strFilename, $strSerialized);
        } catch (Exception $e)
        {
            throw new Exception("Could not open or create $strFilename!");
        }
    }

    /**
     *
     * @param string $strFilename
     * @return Network|InputValue|OutputValue|Values|InterfaceLoadable
     * @uses Exception::__construct()
     * @throws Exception
     */
    public static function loadFromFile(string $strFilename): Network|InputValue|OutputValue|Values|InterfaceLoadable
    {
        if (is_file($strFilename) && is_readable($strFilename))
        {
            $strSerialized = file_get_contents($strFilename);

            if (empty($strSerialized))
                throw new Exception('File ' . basename($strFilename) . ' could not be loaded (file has no object information stored)');

            $objInstance = unserialize($strSerialized);

            if (! ($objInstance instanceof InterfaceLoadable))
                throw new Exception('File ' . basename($strFilename) . ' could not be opened (no ANN format)');

            return $objInstance;
        }

        throw new Exception('File ' . basename($strFilename) . ' could not be opened');
    }
}
